#pragma once

#include "base.h"

typedef unsigned char __u8;
typedef unsigned short __u16;
typedef DWORD __u32;

/*
 *   Magic numbers
*/

/** start address of BIOS segment to scanned for SM-BIOS and DMI-BIOS */
#define BIOS_START_ADDRESS      0xF0000
/** length of the scanned BIOS area for SM-BIOS and DMI-BIOS */
#define BIOS_MAP_LENGTH         0x10000
/** magic 4 bytes to identify SM-BIOS entry point, paragraph boundary */
#define SMBIOS_MAGIC_DWORD      0x5F4D535F /* anchor string "_SM_" */  //BIOS_SM_,UEFI_SM3_
/** magic 4 bytes to identify DMI-BIOS entry point, byte boundary */
#define DMIBIOS_MAGIC_DWORD     0x494d445f /* anchor string "_DMI" */
/** identifier for SM-BIOS structures within SM-BIOS entry point */
#define DMI_STRING              "_DMI_"
/** list of types which are known to have subtyes; expandable! */
#define TYPES_WITH_SUBTYPES     185, 187, 208, 209, 210, 211, 212, 254
/** maximum block size for proc read function */
#define PROC_BLOCK_SIZE         (3*1024)

#pragma pack(1)
struct smbios_entry_point_struct
{
	/** "_SM_", specified as four ASCII characters (5F 53 4D 5F) */
	__u32 anchor_string;
	/** checksum of the Entry Point Structure (EPS). This value, when added to
	 * all other bytes in the EPS, will result in the value 00h (using 8 bit
	 * addition calculations). Values in the EPS are summed starting at offset
	 * 00h, for Entry Point Length bytes.*/
	__u8  entry_point_checksum;
	/** Length of the Entry Point Structure, starting with the Anchor String
	 * field, in bytes, currently 1Fh. */
	__u8  entry_point_length;
	/** identifies the major version of this specification implemented in
	 * the table structures, e.g. the value will be 0Ah for revision 10.22
	 * and 02h for revision 2.1 */
	__u8  major_version;
	/** identifies the minor version of this specification implemented in
	 * the table structures, e.g. the value will be 16h for revision 10.22
	 * and 01h for revision 2.1 */
	__u8  minor_version;
	/** size of the largest SMBIOS structure, in bytes, and encompasses the
	 * structure's formatted area and text strings. This is the value returned
	 * as StructureSize from the Plug-n-Play 'Get SMBIOS Information' function */
	__u16 max_struct_size;
	/** identifies the EPS revision implemented in this structure and identifies
	 * the formatting of offsets 0Bh to 0Fh, one of:
	 * 00h     Entry Point based on SMBIOS 2.1 definition; formatted area is
	 *         reserved and set to all 00h.
	 * 01h-FFh reserved for assignment via this specification */
	__u8  revision;
	/** the value present in the Entry Point Revision field defines the
	 * interpretation to be placed upon these5 bytes. */
	__u8  formated_area[5];
	/** "_DMI_", specified as five ASCII characters (5F 44 4D 49 5F) */
	__u8  intermediate_string[5];
	/** checksum of the Intermediate Entry Point Structure (IEPS). This value,
	 * when added to all other bytes in the IEPS, will result in the value 00h
	 * (using 8 bit addition calculations). Values in the IEPS are summed
	 * starting at offset 10h, for 0Fh bytes */
	__u8  intermediate_checksum;
	/** the 32 bit physical starting address of the read-only SMBIOS Structure
	 * Table, that can start at any 32 bit address. This area contains all of the
	 * SMBIOS structures fully packed together. These structures can then be
	 * parsed to produce exactly the same format as that returned from a 'Get
	 * SMBIOS Structure' function call. */
	__u16 struct_table_length;
	__u32 struct_table_address;
	__u16 no_of_structures;
	__u8  bcd_revision;
};

/** SM-BIOS and DMI-BIOS structure header */
struct smbios_struct
{
	__u8  type;
	__u8  length;
	__u16 handle;
	__u8  subtype;
	/* ... other fields are structure dependend ... */
};

/** DMI-BIOS structure header */
struct dmibios_table_entry_struct
{
	__u16 size;
	__u16 handle;
	__u32 procedure;
};

/** DMI-BIOS entry point structure */
struct dmibios_entry_point_struct
{
	__u8  signature[10];
	__u8  revision;
	struct dmibios_table_entry_struct entry[1];
};

//-------------------
typedef struct
{
	UINT8   Type;
	UINT8   Length;
	UINT8   Handle[2];
} SMBIOS_HEADER, * PSMBIOS_HEADER;

typedef UINT8   SMBIOS_STRING;

typedef struct
{
	SMBIOS_HEADER   Hdr;
	SMBIOS_STRING   Vendor;
	SMBIOS_STRING   BiosVersion;
	UINT8           BiosSegment[2];
	SMBIOS_STRING   BiosReleaseDate;
	UINT8           BiosSize;
	UINT8           BiosCharacteristics[8];
} SMBIOS_TYPE0;

typedef struct
{
	SMBIOS_HEADER   Hdr;
	SMBIOS_STRING   Manufacturer;
	SMBIOS_STRING   ProductName;
	SMBIOS_STRING   Version;
	SMBIOS_STRING   SerialNumber;

	//
	// always byte copy this data to prevent alignment faults!
	//
	GUID			Uuid; // EFI_GUID == GUID?

	UINT8           WakeUpType;
} SMBIOS_TYPE1;

typedef struct
{
	SMBIOS_HEADER   Hdr;
	SMBIOS_STRING   Manufacturer;
	SMBIOS_STRING   ProductName;
	SMBIOS_STRING   Version;
	SMBIOS_STRING   SerialNumber;
} SMBIOS_TYPE2;

typedef struct
{
	SMBIOS_HEADER   Hdr;
	SMBIOS_STRING   Manufacturer;
	UINT8           Type;
	SMBIOS_STRING   Version;
	SMBIOS_STRING   SerialNumber;
	SMBIOS_STRING   AssetTag;
	UINT8           BootupState;
	UINT8           PowerSupplyState;
	UINT8           ThermalState;
	UINT8           SecurityStatus;
	UINT8           OemDefined[4];
} SMBIOS_TYPE3;

//CPU
typedef struct {
	UINT32    ProcessorSteppingId : 4;
	UINT32    ProcessorModel : 4;
	UINT32    ProcessorFamily : 4;
	UINT32    ProcessorType : 2;
	UINT32    ProcessorReserved1 : 2;
	UINT32    ProcessorXModel : 4;
	UINT32    ProcessorXFamily : 8;
	UINT32    ProcessorReserved2 : 4;
} PROCESSOR_SIGNATURE;

typedef struct {
	UINT8    ProcessorVoltageCapability5V : 1;
	UINT8    ProcessorVoltageCapability3_3V : 1;
	UINT8    ProcessorVoltageCapability2_9V : 1;
	UINT8    ProcessorVoltageCapabilityReserved : 1; ///< Bit 3, must be zero.
	UINT8    ProcessorVoltageReserved : 3; ///< Bits 4-6, must be zero.
	UINT8    ProcessorVoltageIndicateLegacy : 1;
} PROCESSOR_VOLTAGE;

typedef struct {
	UINT32    ProcessorFpu : 1;
	UINT32    ProcessorVme : 1;
	UINT32    ProcessorDe : 1;
	UINT32    ProcessorPse : 1;
	UINT32    ProcessorTsc : 1;
	UINT32    ProcessorMsr : 1;
	UINT32    ProcessorPae : 1;
	UINT32    ProcessorMce : 1;
	UINT32    ProcessorCx8 : 1;
	UINT32    ProcessorApic : 1;
	UINT32    ProcessorReserved1 : 1;
	UINT32    ProcessorSep : 1;
	UINT32    ProcessorMtrr : 1;
	UINT32    ProcessorPge : 1;
	UINT32    ProcessorMca : 1;
	UINT32    ProcessorCmov : 1;
	UINT32    ProcessorPat : 1;
	UINT32    ProcessorPse36 : 1;
	UINT32    ProcessorPsn : 1;
	UINT32    ProcessorClfsh : 1;
	UINT32    ProcessorReserved2 : 1;
	UINT32    ProcessorDs : 1;
	UINT32    ProcessorAcpi : 1;
	UINT32    ProcessorMmx : 1;
	UINT32    ProcessorFxsr : 1;
	UINT32    ProcessorSse : 1;
	UINT32    ProcessorSse2 : 1;
	UINT32    ProcessorSs : 1;
	UINT32    ProcessorReserved3 : 1;
	UINT32    ProcessorTm : 1;
	UINT32    ProcessorReserved4 : 2;
} PROCESSOR_FEATURE_FLAGS;
typedef struct {
	PROCESSOR_SIGNATURE        Signature;
	PROCESSOR_FEATURE_FLAGS    FeatureFlags;
} PROCESSOR_ID_DATA;

typedef struct {
	UINT8   AnchorString[4];
	UINT8   EntryPointStructureChecksum;
	UINT8   EntryPointLength;
	UINT8   MajorVersion;
	UINT8   MinorVersion;
	UINT16  MaxStructureSize;
	UINT8   EntryPointRevision;
	UINT8   FormattedArea[5];
	UINT8   IntermediateAnchorString[5];
	UINT8   IntermediateChecksum;
	UINT16  TableLength;
	UINT32  TableAddress;
	UINT16  NumberOfSmbiosStructures;
	UINT8   SmbiosBcdRevision;
} SMBIOS_STRUCTURE_TABLE;

typedef struct {
	//SMBIOS_STRUCTURE_TABLE       Hdr;
	SMBIOS_HEADER    Hdr;
	SMBIOS_STRING    SocketDesignation;
	UINT8                  ProcessorType;         ///< The enumeration value from PROCESSOR_TYPE_DATA.
	UINT8                  ProcessorFamily;       ///< The enumeration value from PROCESSOR_FAMILY_DATA.
	SMBIOS_STRING    ProcessorManufacturer;
	/*SMBIOS_STRING*/unsigned long long      ProcessorId; //by hwb
	SMBIOS_STRING    ProcessorVersion;      
	PROCESSOR_VOLTAGE      Voltage;         
	UINT16                 ExternalClock;   
	UINT16                 MaxSpeed;
	UINT16                 CurrentSpeed;
	UINT8                  Status;
	UINT8                  ProcessorUpgrade;     ///< The enumeration value from PROCESSOR_UPGRADE.
	UINT16                 L1CacheHandle;
	UINT16                 L2CacheHandle;
	UINT16                 L3CacheHandle;
	SMBIOS_STRING    SerialNumber;
	SMBIOS_STRING    AssetTag;
	SMBIOS_STRING    PartNumber;
	UINT8                  CoreCount;
	UINT8                  EnabledCoreCount;
	UINT8                  ThreadCount;
	UINT16                 ProcessorCharacteristics;
	UINT16                 ProcessorFamily2;
	UINT16                 CoreCount2;
	UINT16                 EnabledCoreCount2;
	UINT16                 ThreadCount2;
	UINT16                 ThreadEnabled;
} SMBIOS_TYPE4;

typedef struct _MEMORY_DEVICE_HEADER
{
	SMBIOS_HEADER    Hdr;
	uint16_t physicalArrayHandle;
	uint16_t memoryErrorInformationHandle;
	uint16_t totalWidth;
	uint16_t dataWidth;
	uint16_t size;
	BYTE formFactor;
	BYTE deviceSet;
	BYTE deviceLocator;
	BYTE bankLocator;
	BYTE memoryType;
	uint16_t typeDetail;
	uint16_t speed;
	BYTE manufacturer;
	BYTE serialNumber;
} MEMORY_DEVICE_HEADER, * PMEMORY_DEVICE_HEADER;

#pragma pack()

unsigned char smbios_check_entry_point(void* addr);
struct smbios_entry_point_struct* smbios_find_entry_point(void* base, unsigned long nLen);
struct dmibios_entry_point_struct* dmibios_find_entry_point(void* base, unsigned long nLen);

void handle_smbios_table(void* mapped, unsigned long length);